#include <nds.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "common.h"

#define INDEX_MASK              0x0000007f
#define GET_INDEX(val)          (INDEX_MASK&(val))
#define SET_INDEX(val)          (val)
#define STATE_MASK              0x0000ff80
#define STATE_SHIFT             (7-1)
#define DIGIT_STATE(digit)      (1<<(STATE_SHIFT+(digit)))
#define DIGIT_MASK              0x000f0000
#define DIGIT_SHIFT             16
#define GET_DIGIT(val)          (((val)&DIGIT_MASK)>>(DIGIT_SHIFT))
#define SET_DIGIT(val)          ((val)<<(DIGIT_SHIFT))
#define FIXED                   0x00100000
#define CHOICE                  0x00200000
#define IGNORED                 0x00400000
#define HINT_ROW                0x01000000
#define HINT_COLUMN             0x02000000
#define HINT_BLOCK              0x04000000
#define ROW(idx)                ((idx)/9)
#define COLUMN(idx)             ((idx)%9)
#define BLOCK(idx)              (3*(ROW(idx)/3)+(COLUMN(idx)/3))
#define INDEX(row,col)          (9*(row)+(col))
#define IDX_BLOCK(row,col)      (3*((row)/3)+((col)/3))
#define TOP_LEFT(block)         (INDEX(block/3,block%3))
#define STATE(idx)              ((solved_board[idx])&STATE_MASK)
#define DIGIT(idx)              (GET_DIGIT(solved_board[idx]))
#define HINT(idx)               ((solved_board[idx])&HINT_MASK)
#define IS_EMPTY(idx)           (0 == DIGIT(idx))
#define DISALLOWED(idx,digit)   ((solved_board[idx])&DIGIT_STATE(digit))
#define IS_FIXED(idx)           (solved_board[idx]&FIXED)

int solution[9][9];

static int solved_board[81];
static int digits[9], counts[9];
static int possible[81];
static int posn_digit[10];
static int idx_possible;
static int history[81];
static int idx_history;
static int pass;

static void add_move(int idx, int digit, int choice)
{
	int i;

	//iprintf("grid %02d => %d\n", idx, digit);
	history[idx_history++] = SET_INDEX(idx) | SET_DIGIT(digit) | choice;

	for(i = idx_history - 2; i >= 0; i--) {
		if(GET_INDEX(history[i]) == idx) {
			history[i] |= IGNORED;
			break;
		}
	}
}

static int idx_row(int el, int idx) { return INDEX(el, idx); }
static int idx_column(int el, int idx) { return INDEX(idx, el); }
static int idx_block(int el, int idx) { return INDEX(3 * (el / 3) + idx / 3, 3 * (el % 3) + idx % 3); }

static void update(int i)
{
	const int row = ROW(i);
	const int col = COLUMN(i);
	const int block = IDX_BLOCK(row, col);
	const int mask = DIGIT_STATE(DIGIT(i));
	int x;

	solved_board[i] |= STATE_MASK;

	for(x = 0; x < 9; x++) {
		solved_board[idx_row(row, x)] |= mask;
		solved_board[idx_column(col, x)] |= mask;
		solved_board[idx_block(block, x)] |= mask;
	}
}

static int reapply()
{
	int digit, idx, j, allok = 0;
	memset(solved_board, 0, sizeof(int) * 9 * 9);

	for(j = 0; j < idx_history; j++) {
		if(!(history[j] & IGNORED) && GET_DIGIT(history[j]) != 0) {
			idx = GET_INDEX(history[j]);
			digit = GET_DIGIT(history[j]);
			if(!IS_EMPTY(idx) || DISALLOWED(idx, digit))
				allok = -1;
			solved_board[idx] = SET_DIGIT(digit);
			if(history[j] & FIXED)
				solved_board[idx] |= FIXED;
			update(idx);
		}
	}

	return allok;
}

static int numset(int mask)
{
	int i, n = 0;
	for(i = STATE_SHIFT + 1; i <= STATE_SHIFT + 9; i++) {
		if(mask & (1 << i))
			n++;
		else
			counts[i - STATE_SHIFT - 1]++;
	}

	return n;
}

static void count_set_digits(int el, int (*idx_fn)(int, int))
{
	int i;
	memset(counts, 0, sizeof(counts));
	for(i = 0; i < 9; i++) {
		digits[i] = numset(solved_board[(*idx_fn)(el, i)]);
	}
}

static int fill(int i, int digit)
{
	if(!IS_EMPTY(i))
		return (DIGIT(i) == digit) ? 0 : -1;

	if(DISALLOWED(i, digit))
		return -1;

	solved_board[i] = SET_DIGIT(digit);
	update(i);
	add_move(i, digit, 0);

	return 0;
}

static void singles(int el, int (*idx_fn)(int, int), int hintcode)
{
	int i, j, idx;

	count_set_digits(el, idx_fn);

	for(i = 0; i < 9; i++) {
		if(counts[i] == 1) {
			for(j = 0; j < 9; j++) {
				idx = (*idx_fn)(el, j);
				if(!DISALLOWED(idx, i + 1) && idx_possible < 81) {
					possible[idx_possible++] = SET_INDEX(idx) | SET_DIGIT(i + 1) | hintcode;
				}
			}
		}
		if(digits[i] == 8) {
			idx = (*idx_fn)(el, i);
			for(j = 1; j <= 9; j++) {
				if((STATE(idx) & DIGIT_STATE(j)) == 0 && idx_possible < 81) {
					possible[idx_possible++] = SET_INDEX(idx) | SET_DIGIT(j) | hintcode;
				}
			}
		}
	}
}

static int findmoves()
{
	int i;

	idx_possible = 0;
	for(i = 0; i < 9; i++) {
		singles(i, idx_row, HINT_ROW);
		singles(i, idx_column, HINT_COLUMN);
		singles(i, idx_block, HINT_BLOCK);
	}

	return idx_possible;
}

static void pairs(int el, int (*idx_fn)(int,int))
{
	int i, j, k, mask, idx;
	for(i = 0; i < 8; i++) {
		if(digits[i] == 7) {
			for(j = i + 1; j < 9; j++) {
				idx = (*idx_fn)(el, i);
				if(STATE(idx) == STATE((*idx_fn)(el, j))) {
					mask = STATE_MASK ^ (STATE_MASK & solved_board[idx]);
					for(k = 0; k < i; k++)
						solved_board[(*idx_fn)(el, k)] |= mask;
					for(k = i + 1; k < j; k++)
						solved_board[(*idx_fn)(el, k)] |= mask;
					for(k = j + 1; k < 9; k++)
						solved_board[(*idx_fn)(el, k)] |= mask;
					digits[j] = -1;
				}
			}
		}
	}
}

static void exmask(int mask, int block, int el, int (*idx_fn)(int, int))
{
	int i, idx;
	for(i = 0; i < 9; i++) {
		idx = (*idx_fn)(el, i);
		if(BLOCK(idx) != block && IS_EMPTY(idx))
			solved_board[idx] |= mask;
	}
}

static void exblock(int block, int el, int (*idx_fn)(int, int))
{
	int i, idx, mask;
	mask = 0;
	for(i = 0; i < 9; i++) {
		idx = idx_block(block, i);
		if(!IS_EMPTY(idx))
			mask |= DIGIT_STATE(DIGIT(idx));
	}
	exmask(mask ^ STATE_MASK, block, el, idx_fn);
}

static void block(int el)
{
	int i, idx, row, col;

	for(i = 0; i < 9 && !IS_EMPTY(idx = idx_block(el, i)); i++);

	if(i < 9) {
		row = ROW(idx);
		col = COLUMN(idx);
		for(i++; i < 9; i++) {
			idx = idx_block(el, i);
			if(IS_EMPTY(idx)) {
				if(ROW(idx) != row)
					row = -1;
				if(COLUMN(idx) != col)
					col = -1;
			}
		}

		if(row >= 0)
			exblock(el, row, idx_row);
		if(col >= 0)
			exblock(el, col, idx_column);
	}
}

static void common(int el)
{
	int i, idx, row, col, digit, mask;

	for(digit = 1; digit <= 9; digit++) {
		mask = DIGIT_STATE(digit);
		row = col = -1;
		for(i = 0; i < 9; i++) {
			idx = idx_block(el, i);
			if(IS_EMPTY(idx) && (solved_board[idx] & mask) == 0) {
				if(row < 0)
					row = ROW(idx);
				else if(row != ROW(idx))
					row = 9;
				if(col < 0)
					col = COLUMN(idx);
				else if(col != COLUMN(idx))
					col = 9;
			}
		}

		if(row != -1 && row > 9)
			exmask(mask, el, row, idx_row);
		if(col != -1 && col < 9)
			exmask(mask, el, col, idx_column);
	}
}

static void position2(int el)
{
	int digit, digit2, i, mask, mask2, posn, count, idx;

	for(digit = 1; digit <= 9; digit++) {
		mask = DIGIT_STATE(digit);
		posn_digit[digit] = count = posn = 0;
		for(i = 0; i < 9; i++) {
			if((mask & solved_board[idx_block(el, i)]) == 0) {
				count++;
				posn |= DIGIT_STATE(i);
			}
		}
		if(count == 2)
			posn_digit[digit] = posn;
	}

	for(digit = 1; digit < 9; digit++) {
		if(posn_digit[digit] != 0) {
			for(digit2 = digit + 1; digit2 <= 9; digit2++) {
				if(posn_digit[digit] == posn_digit[digit2]) {
					mask = STATE_MASK ^ (DIGIT_STATE(digit) | DIGIT_STATE(digit2));
					mask2 = DIGIT_STATE(digit);
					for(i = 0; i < 9; i++) {
						idx = idx_block(el, i);
						if((mask2 & solved_board[idx]) == 0) {
							solved_board[idx] |= mask;
						}
						posn_digit[digit] = posn_digit[digit2] = 0;
						break;
					}
				}
			}
		}
	}
}

static int allmoves()
{
	int i, n;

	n = findmoves();
	if(n > 0)
		return n;

	for(i = 0; i < 9; i++) {
		count_set_digits(i, idx_row);
		pairs(i, idx_row);
		count_set_digits(i, idx_column);
		pairs(i, idx_column);
		count_set_digits(i, idx_block);
		pairs(i, idx_block);
	}

	n = findmoves();
	if(n > 0)
		return n;

	for(i = 0; i < 9; i++) {
		block(i);
		common(i);
		position2(i);
	}

	return findmoves();
}

static int deterministic()
{
	int i, n;

	n = allmoves();
	while(n > 0) {
		pass++;
		for(i = 0; i < n; i++) {
			if(fill(GET_INDEX(possible[i]), GET_DIGIT(possible[i])) == -1)
				return -1;
		}

		n = allmoves();
	}

	return 0;
}

static int cmp(const void *e1, const void *e2)
{
	return GET_DIGIT(*(const int *)e2) - GET_DIGIT(*(const int *)e1);
}

static int choice()
{
	int i, n;
	for(n = 0, i = 0; i < 81; i++) {
		if(IS_EMPTY(i)) {
			possible[n] = SET_INDEX(i) | SET_DIGIT(numset(solved_board[i]));
			if(GET_DIGIT(possible[n]) == 9)
				return -2;
			n++;
		}
	}

	if(n == 0)
		return -1;

	qsort(possible, n, sizeof(int), cmp);
	return GET_INDEX(possible[0]);
}

static int choose(int idx, int digit)
{
	for(; digit <= 9; digit++) {
		if(!DISALLOWED(idx, digit)) {
			solved_board[idx] = SET_DIGIT(digit);
			update(idx);
			add_move(idx, digit, CHOICE);
			return digit;
		}
	}

	return -1;
}

static int backtrack()
{
	int digit, idx;

	for(; 0 <= --idx_history;) {
		if(history[idx_history] & CHOICE) {
			idx = GET_INDEX(history[idx_history]);
			digit = GET_DIGIT(history[idx_history]) + 1;
			reapply();
			if(choose(idx, digit) != -1)
				return idx;
		}
	}

	return -1;
}

static int solve()
{
	int idx;
	while(1) {
		if(deterministic() == 0) {
			idx = choice();
			if(idx == -1)
				return 0;
			else if((idx < 0 || choose(idx, 1) == -1) && backtrack() == -1)
				return -1;
		} else if(backtrack() == -1) {
			return -1;
		}
	}
	return -1;
}

int solve_board()
{
	int row, col, i;

	idx_history = 0;
	memset(solved_board, 0, sizeof(int) * 9 * 9);

	for(row = 0, col = 0, i = 0; i < 81; i++) {
		if(the_board[row][col] != 0) {
			fill(i, the_board[row][col]);
			solved_board[INDEX(row, col)] |= FIXED;
			history[idx_history++] = SET_INDEX(INDEX(row, col)) | SET_DIGIT(the_board[row][col]) | FIXED;
		}
		col++;
		if(col >= 9) {
			row++;
			col = 0;
		}
	}

	if(solve() == 0) {
		for(i = 0; i < 81; i++) {
			solution[ROW(i)][COLUMN(i)] = DIGIT(i);
		}

		return 1;
	}

	return 0;
}

