/*
Copyright (C) 1994-1995 Apogee Software, Ltd.
Copyright (C) 2007 JJ Foote

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// udpsetup.c

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>

#if PLATFORM_NDS
#include <nds.h>
#include <dswifi9.h>
#else
#include <arpa/inet.h>
#endif
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>

#include "rt_def.h"
#include "rt_com.h"
#include "rt_util.h"
#include "rt_in.h"
#include "rt_playr.h"
#include "rt_msg.h"
#include "rottnet.h"
#include "rt_main.h"
#include "rt_net.h"
#include "rt_draw.h"
#include "rt_sound.h"

#if PLATFORM_NDS
extern int nds_frame_counter;
#define random() rand()
#endif

/* Define this to use UDP broadcast packets when the server needs to send
 * something to all clients (or to search for new clients).  Leave it
 * undefined to send an individual UDP packet to each client instead. */
//#define USE_UDP_BROADCAST

/* Define this to use non-blocking sockets.  Leave it undefined to use select()
 * with a short timeout to get mostly-non-blocking sockets.
 */
#define USE_NONBLOCKING_SOCKETS

typedef struct
{
	short client;
	short playernumber;
	short command;
	short extra;
	short numplayers;
} setupdata_t;

typedef struct {
	unsigned long ip;
	unsigned port;
} client_box;

static int sock = -1;
#ifdef USE_UDP_BROADCAST
static int bcast_sock = -1;
#endif

// these are stored in network byte order.
int server_ip = 0;
int server_port = 0;

static client_box clients[MAXPLAYERS];
int num_clients = 0;

// Looks up the player number based on the source of a packet.
// NOTE: this precludes running multiple clients on the same machine,
// unless one of them is the server--and even then the server needs to bind to INADDR_ANY
static int find_node(unsigned long ip, unsigned port)
{
	int i;

	// find out which client it came from
	for(i = 0; i < num_clients; i++) {
		if(clients[i].ip == ip && clients[i].port == port)
			return i;
	}

	// try again without matching on port
#if 0
	for(i = 0; i < num_clients; i++) {
		if(clients[i].ip == ip)
			return i;
	}
#endif

	// is it from the server? (FIXME: this shouldn't happen)
	if(ip == INADDR_BROADCAST || ip == server_ip) {
		return 0;
	}

	struct in_addr tempinet;
	tempinet.s_addr = ip;
	printf("Packet from unknown host %s:%d\n", inet_ntoa(tempinet), ntohs(port));

	return -2;
}

static struct sockaddr_in remote_addr;
static int remote_addr_len;

/* Attempts to read an incoming UDP packet.
 * If no packet is available or there was an error, returns -1
 * If packet is from an unknown host, returns -2
 * Otherwise, returns the player number of the client who sent it
 * In any case, fills the "rottcom" structure with the packet's payload.
 */
void ReadUDPPacket()
{
	int flags, i;

	if(sock == -1) {
		// not connected
		rottcom->remotenode = -1;
		return;
	}

#ifdef USE_NONBLOCKING_SOCKETS
	flags = 0;
	remote_addr_len = sizeof(remote_addr);

	i = recvfrom(sock, rottcom->data, MAXPACKETSIZE, flags, &remote_addr, &remote_addr_len);
	if(i == -1) {
		rottcom->datalength = 0;
		rottcom->remotenode = -1;
	} else {
		rottcom->datalength = i;
		rottcom->remotenode = find_node(remote_addr.sin_addr.s_addr, remote_addr.sin_port);
	}
#else
	struct timeval tv;
	fd_set fd;

	FD_ZERO(&fd);
	FD_SET(sock, &fd);
	tv.tv_sec = 0;
	tv.tv_usec = 1000 * 10; // max wait: 10ms

	i = select(sock + 1, &fd, NULL, NULL, &tv);
	if(i == -1) {
		perror("Error trying to read from socket");
		rottcom->datalength = 0;
		rottcom->remotenode = -1;
	} else if(i > 0 && FD_ISSET(sock, &fd)) {
		// packet ready; slurp it up
		rottcom->datalength = recvfrom(sock, rottcom->data, MAXPACKETSIZE, flags, &remote_addr, &remote_addr_len);
		rottcom->remotenode = find_node(remote_addr.sin_addr.s_addr, remote_addr.sin_port);
		//printf("Got a packet from %s, port %d\n", inet_ntoa(remote_addr.sin_addr), ntohs(remote_addr.sin_port));
	} else {
		// no packet received yet
		rottcom->datalength = 0;
		rottcom->remotenode = -1;
	}
#endif
}

/* Sends a UDP packet.
 * The "rottcom" structure must be filled with the packet's payload,
 * including the player number of the designated recipient of the packet.
 */
void WriteUDPPacket()
{
	int flags, i;

	if(sock == -1) {
		// not connected
		return;
	} else if(rottcom->remotenode == -1) {
		// bogus recipient
		return;
	}

	flags = 0;
	remote_addr_len = sizeof(remote_addr);

#ifdef USE_UDP_BROADCAST
	if(rottcom->remotenode == MAXNETNODES) {
		i = send(bcast_sock, rottcom->data, rottcom->datalength, flags);
	} else
#endif
   	{
		remote_addr.sin_family = AF_INET;
		remote_addr.sin_addr.s_addr = clients[rottcom->remotenode].ip;
		remote_addr.sin_port = clients[rottcom->remotenode].port;

		i = sendto(sock, rottcom->data, rottcom->datalength, flags, &remote_addr, remote_addr_len);
	}

	if(i <= -1) {
		perror("Failed to send packet");
	}
}

#if PLATFORM_NDS
// timer IRQ callback for dswifi library
static void timer_50ms(void)
{
	Wifi_Timer(50);
}

// CPU sync function for dswifi library
static void arm9_synctoarm7(void)
{
	REG_IPC_FIFO_TX = 0x87654321;
}

// Re-instates the TIMER3 IRQ.  Call this after SDL inits
// to keep WiFi working properly.
static int timer3_irq_needs_fixing = 0;
void fix_timer3_irq(void)
{
	if(timer3_irq_needs_fixing) {
		irqSet(IRQ_TIMER3, timer_50ms);
		irqEnable(IRQ_TIMER3);
		timer3_irq_needs_fixing = 1;
	}
}

#endif

/* Initializes networking.
 * This function must be called prior to any network communications,
 * but after determining whether we are the server or a client,
 * and the server's IP and port.  Also performs any additional
 * platform-specific network initialization that may be necessary.
 */
void network_init(void)
{
	struct sockaddr_in my_addr;
#ifdef USE_UDP_BROADCAST
	struct sockaddr_in bcast_addr;
#endif
	int one = 1, flags;

#ifdef USE_UDP_BROADCAST
	if(sock > -1 || bcast_sock > -1)
#else
	if(sock > -1)
#endif
   	{
		printf("Closing socket first\n");
		if(sock > -1) close(sock);
#ifdef USE_UDP_BROADCAST
		if(bcast_sock > -1) close(bcast_sock);
#endif
	}

#if PLATFORM_NDS
	// initialize WiFi
	{
		iprintf("Initializing WiFi...");
		// tell dswifi to use a 64KB heap (default is 128KB), because we're barely squeaking by on RAM.
		u32 wifi_pass = Wifi_Init(WIFIINIT_OPTION_USELED | WIFIINIT_OPTION_USEHEAP_64);
		REG_IPC_FIFO_TX = 0x12345678;
		REG_IPC_FIFO_TX = wifi_pass;

		if(!rottnet_settings_from_menu.valid) {
			// XXX: When SDL initializes, it'll call irqInit() again, which will erase
			// or handler, so we'll need to call irqSet() and irqEnable() again later.
			irqInit();
			timer3_irq_needs_fixing = 1;
		}

		// Setup a Timer to call the Wifi update function every 50ms
		TIMER3_CR = 0;
		irqSet(IRQ_TIMER3, timer_50ms);
		irqEnable(IRQ_TIMER3);
		Wifi_SetSyncHandler(arm9_synctoarm7);
		TIMER3_DATA = -6553;
		TIMER3_CR = TIMER_ENABLE | TIMER_IRQ_REQ | TIMER_DIV_256;

		iprintf(" ARM9");
		while(Wifi_CheckInit() == 0) { }
		iprintf(" ARM7.\n");
	}

	// connect to an AP
	{
		int wifi_status;
		iprintf("Connecting via WFC data...");
		Wifi_AutoConnect();
		while(1) {
			wifi_status = Wifi_AssocStatus();
			if(wifi_status == ASSOCSTATUS_ASSOCIATED) {
				iprintf(" done\n");
				break;
			} else if(wifi_status == ASSOCSTATUS_CANNOTCONNECT) {
				iprintf(" FAILED\n");
				return;
			}
		}
	}
#endif

	// create a UDP socket
	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if(sock <= -1) {
		perror("Failed to open socket");
		return;
	}

#ifdef USE_NONBLOCKING_SOCKETS
#if PLATFORM_NDS
	flags = 1;
	if(ioctl(sock, FIONBIO, &flags)) {
		perror("Failed to set non-blocking socket");
		return;
	}
#else
	flags = fcntl(sock, F_GETFL, 0);
	if(flags <= -1) {
		perror("Failed to query socket for non-blocking mode");
		return;
	}
	flags |= O_NONBLOCK;
	if(fcntl(sock, F_SETFL, flags) <= -1) {
		perror("Failed to set non-blocking socket");
		return;
	}
#endif
#endif

	if(!rottcom->client) {
		// bind the socket to a local interface
		memset(&my_addr, 0, sizeof(my_addr));
		my_addr.sin_family = AF_INET;
		my_addr.sin_addr.s_addr = server_ip ? server_ip : htonl(INADDR_ANY);
		my_addr.sin_port = server_port;

		if(bind(sock, &my_addr, sizeof(my_addr)) <= -1) {
			perror("Failed to bind socket");
			return;
		}
	}

#ifdef USE_UDP_BROADCAST
	if(setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (char *) &one, sizeof(one)) <= -1) {
		perror("Failed to set broadcast flag on socket");
		return;
	}

	if(!rottcom->client) {
		// for the server, create an additional socket that binds to the broadcast address,
		// so we can send out UDP broadcast packets to advertise, or send info to all clients.
		bcast_sock = socket(AF_INET, SOCK_DGRAM, 0);
		if(bcast_sock <= 1) {
			perror("Failed to open broadcast socket");
			return;
		}

		memset(&bcast_addr, 0, sizeof(bcast_addr));
		bcast_addr.sin_family = AF_INET;
		bcast_addr.sin_addr.s_addr = htonl(INADDR_ANY);
		bcast_addr.sin_port = htons(0);
		if(bind(bcast_sock, &bcast_addr, sizeof(bcast_addr)) <= -1) {
			perror("Failed to bind broadcast socket");
			return;
		}

		if(setsockopt(bcast_sock, SOL_SOCKET, SO_BROADCAST, (char *) &one, sizeof(one)) <= -1) {
			perror("Failed to set broadcast flag on broadcast socket");
			return;
		}

		memset(&bcast_addr, 0, sizeof(bcast_addr));
		bcast_addr.sin_family = AF_INET;
		bcast_addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);
		bcast_addr.sin_port = htons(server_port);
		if(connect(bcast_sock, &bcast_addr, sizeof(bcast_addr)) <= -1) {
			perror("Failed to connect broadcast socket");
			return;
		}
	}
#endif

	if(rottcom->client)
		printf("Networking enabled.  I'm a client.\n");
	else
		printf("Networking enabled.  I'm the server.  %s:%d\n", inet_ntoa(my_addr.sin_addr), ntohs(my_addr.sin_port));
}

/* Disables networking.
 * Stops all network communications.  After this, you must call
 * network_init() again to re-enable networking.
 */
void network_uninit(void)
{
#ifdef USE_UDP_BROADCAST
	if(sock > -1 || bcast_sock > -1)
#else
	if(sock > -1)
#endif
   	{
		printf("Closing socket\n");
		if(sock > -1) {
			shutdown(sock, 2);
			close(sock);
		}
#ifdef USE_UDP_BROADCAST
		if(bcast_sock > -1) {
			shutdown(bcast_sock, 2);
			close(bcast_sock);
		}
#endif
	}

#if PLATFORM_NDS
	Wifi_DisableWifi();
	timer3_irq_needs_fixing = 1;
#endif
}

#define MAXNETMESSAGES 25
static int messagesused;
static boolean StringUsed[MAXNETMESSAGES];
static char * NetStrings[MAXNETMESSAGES]=
{
	"\nUm, I'm sure Player %d will join us soon.\n",
	"\nPlayer %d is starting to tick me off.\n",
	"\nIt seems Player %d has gone for a Moon Pie.\n",
	"\nHow long can Player %d take setting the dang thing up?\n",
	"\nPlayer %d...where are you?\n",
	"\nSigh.  Player %d is a toadie.\n",
	"\nGo give Player %d a good swift kick in the...head.\n",
	"\nIs Player %d running off a removable drive or something?\n",
	"\nPlayer %d is a popo-head.\n",
	"\nPlayer %d is attempting to escape off the map.\n",
	"\nPLAYER %d!!!! GOON!\n",
	"\nYou know, waiting for Player %d reminds me of a story....\n",
	"\nTwo Player %d's walk into a bar....\n",
	"\nHow many Player %d's does it take to change a lightbulb?\n"
		"None, 'cause they don't DO anything.  They just SIT there.\n",
	"\nANY TIME NOW PLAYER %d!!!\n",
	"\nI hear Player %d sucks dog's toes.\n",
	"\nWho votes that Player %d gets left outta this game (y/n)?\n",
	"\nLunch break's over, Player %d!\n",
	"\nWe're all waiting, Player %d....  Hello, McFly!!!\n",
	"\nNumber %d Player, our Number 1 delayer....\n",
	"\nINSTRUCTIONS: Player %d runs the setup program....\n",
	"\nOnce Player %d deigns to join us, let's toast 'em.\n",
	"\nPssst...If you go wake up Player %d, I'll put you in God mode.\n",
	"\nOkay, when we start, I'm giving Player %d only 5 hit points.\n",
	"\nIs Player %d in Shrooms Mode or what?\n"
};

/*
===================
=
= SetupRandomMessages
=
===================
*/
static void SetupRandomMessages ( void )
{
	messagesused=0;
	memset(StringUsed,0,sizeof(StringUsed));
#if PLATFORM_NDS
	srand(nds_frame_counter);
#elif PLATFORM_UNIX
	srand(time(NULL));
#else
	//randomize();
#endif
}

/*
===================
=
= PrintRandomMessage
=
===================
*/
static void PrintRandomMessage (int message, int player)
{
	printf(NetStrings[message],player);
}

/*
===================
=
= GetRandomMessage
=
===================
*/
static int GetRandomMessage ( void )
{
	boolean found=false;
	int num;

	if (messagesused==MAXNETMESSAGES)
		SetupRandomMessages();

	while (found==false)
	{
		num = random() % MAXNETMESSAGES;
		if (StringUsed[num]==false)
		{
			StringUsed[num]=true;
			found=true;
			messagesused++;
		}
	}
	return num;
}

#define client_NoResponse 0
#define client_Echoed 1
#define client_Done 2

static int clientstate[MAXNETNODES+1];
/*
===================
=
= ResetNodeAddresses
=
= Finds all the nodes for the game and works out player numbers among them
=
===================
*/
static void ResetNodeAddresses ( void )
{
	// Zero out client state structure
	num_clients = 0;

	clientstate[0]=client_Done;
}

/*
===================
= UDPSendPacket
===================
*/
static void SendPacket(int destination)
{
#ifndef USE_UDP_BROADCAST
	int i;

	if(destination == MAXNETNODES) {
		i = (rottcom->consoleplayer ? 2 : 1);
		for(; i < num_clients; i++) {
			rottcom->remotenode = i;
			WriteUDPPacket();
		}
	} else
#endif
   	{
		rottcom->remotenode = destination;
		WriteUDPPacket();
	}
}

/*
===================
= UDPGetPacket
===================
*/
static int GetPacket(void)
{
	ReadUDPPacket();
	if(rottcom->remotenode == -2) {
		// packet from unknown source
		rottcom->remotenode = -1;
		return 1;
	} else if(rottcom->remotenode == -1) {
		// no packet
		return 0;
	} else {
		// yay packet
		return 1;
	}
}


/*
===================
=
= LookForNodes
=
= Finds all the nodes for the game and works out player numbers among them
=
===================
*/

static setupdata_t nodesetup;

#define cmd_FindClient 1
#define cmd_HereIAm 2
#define cmd_YouAre 3
#define cmd_IAm 4
#define cmd_AllDone 5
#define cmd_Info 6

#define MAXWAIT 10

void LookForNodes (void)
{
#if !PLATFORM_NDS
	time_t			now;
#endif
	int             i;
	int             oldsec;
	int				last_said_hi = 0;
	setupdata_t     *setup;
	boolean         done;
	short           numplayers,extra;
	boolean         showednum;
	short           secondsleft;
	boolean         masterset=false;

	network_init();

	done=false;

	oldsec = -1;
	secondsleft = MAXWAIT;
	setup = (setupdata_t *)&rottcom->data;
	showednum = false;
	SetupRandomMessages ();

	if (!rottcom->client)
	{
		printf("SERVER MODE:\n");
		printf("============\n");
		printf("Attempting to find all clients for %i player NETROTT\n", rottcom->numplayers);

		nodesetup.client=0;
		rottcom->client=0;
		ResetNodeAddresses ();

/*
		// try to find our real IP (in case we bound to INADDR_ANY or something)
		struct sockaddr_in tempaddr;
		int sockaddr_len = sizeof(struct sockaddr_in);
		if(getsockname(sock, &tempaddr, &sockaddr_len)) {
			printf("Hey there, %u is now %u!\n", server_ip, tempaddr.sin_addr.s_addr);
			server_ip = tempaddr.sin_addr.s_addr;
		} else {
			perror("getsockname() failed");
		}
*/
		// add ourselves as the zero'th player
		clients[num_clients].ip = server_ip;
		clients[num_clients].port = server_port;
		num_clients++;

		if (rottcom->consoleplayer) {
			masterset=true;

			// add ourselves as the first player too, since we're not a dedicated server
			clients[num_clients].ip = server_ip;
			clients[num_clients].port = server_port;
			clientstate[num_clients]=client_Done;
			num_clients++;
		}
	}
	else
	{
		printf("CLIENT MODE:\n");
		printf("============\n");
		printf("Attempting to find server\n");

		rottcom->numplayers=MAXPLAYERS;
		nodesetup.client=1;
		rottcom->client=1;

		// add ourselves as the zero'th player
		clients[num_clients].ip = 0; /* FIXME */
		clients[num_clients].port = 0;
		num_clients++;

		// add the server as the first player
		clients[num_clients].ip = server_ip;
		clients[num_clients].port = server_port;
		num_clients++;
	}

	// build local setup info

	nodesetup.playernumber = 0;
#if PLATFORM_NDS
	oldsec = nds_frame_counter / 60;
#else
	time(&now);
	oldsec = now;
#endif

	do
	{
		// check for aborting
		/*
		while ( bioskey(1) )
		{
			if ( (bioskey (0) & 0xff) == 27)
				Error ("\n\nNetwork game synchronization aborted.");
		}
		*/
		if (rottcom->client) // Client code
		{
			// say hi once every 3 seconds if we're still not connected
#if PLATFORM_NDS
			if(nds_frame_counter / 60 - last_said_hi >= 3)
#else
			time(&now);
			if(now - last_said_hi >= 3)
#endif
			{
				nodesetup.command=cmd_HereIAm;
				if (!rottcom->consoleplayer)
					nodesetup.extra=1;
				else
					nodesetup.extra=0;
				memcpy (&rottcom->data, &nodesetup,sizeof(*setup));
				rottcom->datalength = sizeof(*setup);
				SendPacket (1);     // send to server
#if PLATFORM_NDS
				last_said_hi = nds_frame_counter / 60;
#else
				time(&now);
				last_said_hi = now;
#endif
			}

			// listen to the network
			while (GetPacket ())
			{
				extra      = setup->extra;
				numplayers = setup->numplayers;

				/*
				if (remotetime != -1)
				{    // an early game packet, not a setup packet
					if (rottcom->remotenode == -1)
						Error ("\n\nUnkown game packet: Other ROTT server present");
				}
				*/

				if (setup->client==0) // It's a server packet
				{
					switch (setup->command)
					{
/*
						case cmd_FindClient:
							//printf("Server is looking for clients\n");
							if (rottcom->remotenode==-1) // only set if it is an unknown pkt
							{
								// copy the server's address
								clients[1].ip = remote_addr.sin_addr.s_addr;
								clients[1].port = remote_addr.sin_port;
							}
							nodesetup.command=cmd_HereIAm;
							if (!rottcom->consoleplayer)
								nodesetup.extra=1;
							else
								nodesetup.extra=0;
							memcpy (&rottcom->data, &nodesetup,sizeof(*setup));
							rottcom->datalength = sizeof(*setup);
							SendPacket (1);     // send to server
							printf (".");
							break;
*/
						case cmd_Info:
							//printf("Server is telling us something\n");
							if (showednum==false)
							{
								printf("\nServer is looking for %d players\n",numplayers);
								showednum = true;
							}
							if ((extra>>8)!=0)
							{
								if ((extra>>8)==100) {
									printf("\nServer found player %d\n",(extra&0xff));
								} else {
									PrintRandomMessage((extra>>8)-1,(extra&0xff));
								}	
							}
							break;
						case cmd_YouAre:
							printf("Server is acknowledging our join request\n");
							rottcom->consoleplayer=setup->playernumber;
							nodesetup.command=cmd_IAm;
							nodesetup.playernumber=setup->playernumber;
							memcpy (&rottcom->data, &nodesetup,sizeof(*setup));
							rottcom->datalength = sizeof(*setup);
							SendPacket (1);     // send to server
							//printf (".");
							printf ("\nI am player %d\n",setup->playernumber);
							break;
						case cmd_AllDone:
							printf("Server is ready to begin the game\n");
							rottcom->numplayers=setup->playernumber;
							done=true;
							//printf ("!");
							break;
					}
				} else {
					printf("Got a client-originated packet\n");
				}
			}
		}
		else // It's the server
		{
			// listen to the network
			while (GetPacket ())
			{
				extra      = setup->extra;
				/*
				if (remotetime != -1)
				{    // an early game packet, not a setup packet
					if (rottcom->remotenode == -1)
						Error ("\n\nUnkown game packet: Other clients still in game.");
				}
				*/
				if (setup->client==1) // It's a client packet
				{
					switch (setup->command)
					{
						case cmd_HereIAm:
							{
								int pnum;

								if ((masterset==false) && (extra==1)) // some client is using the 'master' param
								{
									ResetNodeAddresses ();

									// add ourself back in as the zero'th client
									clients[num_clients].ip = server_ip;
									clients[num_clients].port = server_port;
									num_clients++;

									masterset=true;
									rottcom->remotenode=-1;
								}

								pnum=num_clients;
								if (rottcom->remotenode==-1) // only set if it is an unknown pkt
								{
									// copy the client's address
									if(num_clients < MAXPLAYERS) {
										clients[num_clients].ip = remote_addr.sin_addr.s_addr;
										clients[num_clients].port = remote_addr.sin_port;
										num_clients++;
										printf("Hello, %s:%d!\n", inet_ntoa(remote_addr.sin_addr), ntohs(remote_addr.sin_port));
									} else {
										printf("Sorry %s:%d, you can't connect because I'm already full.\n", inet_ntoa(remote_addr.sin_addr), ntohs(remote_addr.sin_port));
										continue;
									}

									// let everyone else know we've got a new player
									printf ("\nFound Player %d\n",pnum);
									nodesetup.extra=(short)(100<<8);
									nodesetup.extra |= pnum;
									nodesetup.command=cmd_Info;
									nodesetup.numplayers=rottcom->numplayers;
									rottcom->datalength = sizeof(*setup);
									memcpy (&rottcom->data, &nodesetup,sizeof(*setup));
									SendPacket (MAXNETNODES);     // send to all
								}
								else
									pnum=rottcom->remotenode;
								if (clientstate[pnum]>client_Echoed)
									continue;
								nodesetup.command=cmd_YouAre;
								nodesetup.playernumber=pnum;
								memcpy (&rottcom->data, &nodesetup,sizeof(*setup));
								SendPacket (pnum);   // send back to client
								clientstate[pnum]=client_Echoed;
							}
							break;
						case cmd_IAm:
							if (rottcom->remotenode==-1) // Shouldn't happen
								continue;
							//                      Error("\n\nReceived Identity packet before identification\n");
							if (rottcom->remotenode!=setup->playernumber)
								Error("\n\nReceived Incorrect player number\n");
							printf ("Finished Player %d\n",rottcom->remotenode);
							clientstate[rottcom->remotenode]=client_Done;
							break;
					}
				}
				else
				{
					/*
					if (rottcom->remotenode!=0)
						Error("\n\nMultiple ROTT Servers!\n");
					*/
					switch (setup->command)
					{
						case cmd_Info:
							if (((extra>>8)!=0) && (rottcom->consoleplayer))
							{
								if ((extra>>8)==100)
									printf("\nServer found player %d\n",(extra&0xff));
								else
									PrintRandomMessage((extra>>8)-1,(extra&0xff));
							}
							break;
					}
				}
			}

			// Check to see if we are indeed done
			done=true;
			if (num_clients > rottcom->numplayers+1)
				Error("\n\nFound too many players=%d\n",num_clients);
			if (num_clients <= rottcom->numplayers) {
				//printf("Still waiting on %d players\n", rottcom->numplayers + 1 - num_clients);
				done=false;
			}
			else
			{
				for (i=1;i<num_clients;i++)
					if (clientstate[i]!=client_Done) {
						//printf("Still waiting on player %d\n", i);
						done=false;
					}
			}

#if PLATFORM_NDS
			if (nds_frame_counter / 60 == oldsec)
				continue;
#else
			time(&now);
			if(oldsec == now)
				continue;
#endif

			secondsleft--;
			if (secondsleft==0)
			{
				secondsleft=MAXWAIT;
				nodesetup.extra=(GetRandomMessage())+1;
				nodesetup.extra<<=8;
				nodesetup.extra|=num_clients;
				nodesetup.command=cmd_Info;
				nodesetup.numplayers=rottcom->numplayers;
				rottcom->datalength = sizeof(*setup);
				memcpy (&rottcom->data, &nodesetup,sizeof(*setup));

				SendPacket (MAXNETNODES);     // send to all
			}

#if PLATFORM_NDS
			oldsec = nds_frame_counter / 60;
#else
			time(&now);
			oldsec = now;
#endif
			//printf (".");

/*
			// Make a general inquiry for any potential ROTT players
			nodesetup.extra=0;
			nodesetup.command=cmd_FindClient;
			nodesetup.numplayers=rottcom->numplayers;
			rottcom->datalength = sizeof(*setup);
			memcpy (&rottcom->data, &nodesetup,sizeof(*setup));

			SendPacket (MAXNETNODES);     // send to all
*/
		}

	} while (done==false);

	// Done with finding players send out startup info

	if (!rottcom->client)
	{
/*
		int otime;

#if PLATFORM_NDS
		oldsec = nds_frame_counter / 60 + 1;
#else
		time(&now);
		oldsec = now + 1;
#endif
		if (oldsec>59)
			oldsec-=60;
#if PLATFORM_NDS
		otime = nds_frame_counter / 60 - 1;
		while(oldsec != nds_frame_counter / 60)
#else
		time(&now);
		otime = now - 1;
		while(oldsec != now)
#endif
		{
#if PLATFORM_NDS
			if (nds_frame_counter / 60 == otime)
				continue;
#else
			time(&now);
			if(now == otime)
				continue;
#endif
#if PLATFORM_NDS
			otime = nds_frame_counter / 60;
#else
			otime = now;
#endif
*/
			nodesetup.command=cmd_AllDone;
			nodesetup.playernumber=num_clients-1;
			rottcom->datalength = sizeof(*setup);
			memcpy (&rottcom->data, &nodesetup,sizeof(*setup));

			SendPacket (MAXNETNODES);     // send to all
/*
		}
*/

		if (rottcom->consoleplayer)
			printf ("Server is player %i of %i\n", rottcom->consoleplayer, rottcom->numplayers);
		else
			printf ("Server is standalone\n");
	}
	else
		printf ("\nConsole is player %i of %i\n", rottcom->consoleplayer, rottcom->numplayers);

	while (GetPacket ()) {}
}


