/*
Copyright (C) 2007 JJ Foote

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include <nds.h>
#include <stdio.h>
#include <string.h>
#include <dswifi9.h>
#include "rt_def.h"
#include "fx_man.h"
#include "music.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// LOW LEVEL SOUND FUNCTIONS
//////////////////////////////////////////////////////////////////////////////////////////////////////////

// these structs must match the definitions on the ARM7 side
typedef struct
{
	const void *data;
	u32 len;
	u8 vol;
	u8 pan;
	u8 loop;
	u8 state;
	u16 sample_rate;
	u8 bit_rate;
	u8 priority;
} channel_t;

typedef struct
{
	channel_t channels[16];
	u8 master_volume;
	u8 PADDING[3];
} soundsys_t;

static soundsys_t cache_sndsys;
static soundsys_t *sndsys = (soundsys_t *) ((int)&cache_sndsys + 0x400000); // point at the uncached mirror of RAM

static int nds_volume = 64;
static int nds_music_volume = 64;

#define MUSIC_SAMPLE_RATE 22050 /* This must match the actual sample rate of the sound files */
#define MAX_BUFFER_SIZE (0x10000 * 2) /* Buffers larger than this introduce artifacts, because the ARM9 can't fill it fast enough */

FILE *music_fp;
bool music_needs_rebuffering = false;
char music_buffer[MAX_BUFFER_SIZE];
int music_buffer_len = MAX_BUFFER_SIZE;
char *music_ptr = (char *) ((int)&music_buffer + 0x400000); // point at the uncached mirror of RAM

static void channel_stop(int c); // prototype (defined here)

#ifndef NO_SOFTWARE_ADPCM_DECODING
void decode_adpcm_block(); // prototype (defined in adpcm_decoder.c)

int adpcm_decode_bytes_to_read;
int adpcm_decode_bytes_read;
char *adpcm_decode_ptr;
#endif

// finds an available sound channel
static int find_free_channel(int priority)
{
	static int channel_to_steal = 0;
	int i;
	channel_t *c;

	// see if there's a free channel we can use
	for(i = 0, c = sndsys->channels; i < 15; i++, c++) {
		if(!c->state) {
			//iprintf("Using channel %d\n", i);
			return i;
		}
	}

	// all channels are in use; try to find one at a lower priority than us, that we can steal
	// rotate around the list, so we don't always snipe the same channel (poor guy)
	for(i = channel_to_steal, c = sndsys->channels; i < 15; i++, c++) {
		if(c->priority <= priority) {
			//iprintf("Bumping channel %d\n", i);
			channel_to_steal++;
			channel_to_steal = (channel_to_steal == 14 ? 0 : channel_to_steal + 1);
			return i;
		}
	}
	for(i = channel_to_steal - 1, c = sndsys->channels; i >= 0; i++, c++) {
		if(c->priority <= priority) {
			//iprintf("Bumping channel %d\n", i);
			channel_to_steal++;
			channel_to_steal = (channel_to_steal == 14 ? 0 : channel_to_steal + 1);
			return i;
		}
	}

//#define SOUND_CHANNEL_BULLY
#ifdef SOUND_CHANNEL_BULLY
	// no channels are available; let's steal one anyway, even though it's a higher priority than us
	i = channel_to_steal;
	//iprintf("Stealing channel %d\n", i);
	channel_stop(i);
	channel_to_steal = (channel_to_steal == 14 ? 0 : channel_to_steal + 1);
	return i;
#endif

	// no channels are available; give up
	//iprintf("No channel available\n");
	return -1;
}

// starts playing a sample on the specified channel
static void channel_start_sound(int c, char *data, int len, int volume, int pan, int loop, int sample_rate, int bit_rate, int priority)
{
	channel_t *ch;

	if(c < 0)
		return;

	ch = &sndsys->channels[c];
	ch->data = data;
	ch->len = len;
	ch->pan = pan;
	ch->vol = volume;
	ch->loop = loop;
	ch->sample_rate = sample_rate;
	ch->bit_rate = bit_rate;
	ch->priority = abs(priority);
	ch->state = (priority < 0 ? 6 : 1); // if priority == -1, consider this a streaming music channel

	DC_FlushRange(data, len);
}

// stops playing sound on the specified channel
static void channel_stop(int c)
{
	if(c < 0)
		return;

	sndsys->channels[c].state = 3;
}

// tests if the specified channel is done playing its sample
static int channel_done_playing(int c)
{
	if(c < 0)
		return 1;

	//return sndsys->channels[c].state == 2 || sndsys->channels[c].state == 0;
	return sndsys->channels[c].state == 0;
}

// change the volume on a sound channel
static void channel_change_volume(int c, int volume)
{
	channel_t *ch;

	if(c < 0)
		return;

	ch = &sndsys->channels[c];
	if(ch->state == 2 || ch->state == 4) {
		ch->vol = volume;
		ch->state = 4;
	}
}

// change the panning on a sound channel
static void channel_change_pan(int c, int pan)
{
	channel_t *ch;

	if(c < 0)
		return;

	ch = &sndsys->channels[c];
	if(ch->state == 2 || ch->state == 4) {
		ch->pan = pan;
		ch->state = 4;
	}
}

// stop playing sound on all channels
static void stop_all_sounds(void)
{
	int i;
	channel_t *c;

	for(i = 0, c = sndsys->channels; i < 16; i++, c++) {
		if(c->state)
			c->state = 3;
	}
}

// initialize sound system
static int sound_init(void)
{
	// pass the address of our sound structure to the ARM7 via the FIFO
	//DC_FlushRange(sndsys, 4);
	REG_IPC_FIFO_TX = (int) sndsys;
	return 0;
}

// update master volume
static void update_master_volume(void)
{
	sndsys->master_volume = nds_volume;
}

// calcluate volume and pan values from a given angle and distance
static void calculate_volume_and_panning(int angle, int distance, int *vol, int *pan)
{
	int left, right, mypan;

	if(abs(distance) > 255 || angle < 0 || angle > 31) {
		// too far away to hear anything, or the angle is bogus
		//iprintf("GG: dist=%d, ang=%d\n", distance, angle);
		*vol = 0;
		*pan = 0;
		return;
	}

	if(distance < 0) {
		// if it's behind us, translate (wrap around the angle)
		distance = -distance;
		angle += 16;
	}

	// quick and dirty calculation of the PanTable array used by the old audiolib
	if(angle <= 15) {
		right = 255 * (64 - distance / 4) / 64;
		left = right - (right * angle) / 8;
	} else {
		left = 255 * (64 - distance / 4) / 64;
		right = left - (left * angle) / 8;
	}

	if(right == 0)
		mypan = 127;
	else
		mypan = 64 * left / right;

	if(mypan > 127) mypan = 127;
	if(mypan < 0) mypan = 0;

	*vol = (255 - distance) >> 1;
	*pan = mypan;
}

// the DS only likes signed PCM data.  Sound usually comes in 16-bit signed, or 8-bit unsigned.
// This function will convert 8-bit unsigned to 8-bit signed, so the DS will like it better.
static inline void unsigned_to_signed(char *buf, int len)
{
	int i;
	for(i = 0; i < len; i++)
		buf[i] ^= 0x80;
}

// decodes one ADPCM block from the stream, and adds it to the music buffer
void buffer_one_adpcm_block(int *bytes_read, int bytes_to_read, char **ptr)
{
	extern s16 *adpcm_buffer;
	extern int adpcm_buffer_len;
	int i;

	decode_adpcm_block();

	i = (adpcm_buffer_len < (bytes_to_read - *bytes_read) ? adpcm_buffer_len : (bytes_to_read - *bytes_read));
	memcpy(*ptr, adpcm_buffer, i);

	*ptr += i;
	*bytes_read += i;
}

// buffers some music data
void buffer_more_music()
{
	const int channel = 15;
	int bytes_to_read;
	int bytes_read;
	char *ptr;
	channel_t *ch;

	if(music_fp == NULL) {
		return;
	}

	ch = &sndsys->channels[channel];

	switch(ch->state) {
		case 0: // initial load; fill entire buffer
		case 3:
			ptr = (char *) ((int)&music_buffer[0] + 0x400000);
			bytes_to_read = music_buffer_len;
			break;
		case 7: // fill top half of buffer
			ptr = (char *) ((int)&music_buffer[0] + 0x400000);
			bytes_to_read = music_buffer_len / 2;
			break;
		case 8: // fill bottom half of buffer
			ptr = (char *) ((int)&music_buffer[music_buffer_len / 2] + 0x400000);
			bytes_to_read = music_buffer_len / 2;
			break;
		default:
			return;
	};

	ch->state = 2;

// Option 1: software decode the ADPCM on the ARM9, and stream raw S16 data to the ARM7.
// Option 2: just stream the ADPCM to the ARM7.  This causes glitches whenever the buffer loops
// 			 around, because the ARM7 sound hardware resets the decoding state upon repeat.
#ifndef NO_SOFTWARE_ADPCM_DECODING
	extern exit_t playstate;
	adpcm_decode_bytes_read = 0;
	adpcm_decode_bytes_to_read = 0;
	adpcm_decode_ptr = NULL;

	bytes_read = 0;
	if(playstate == ex_stillplaying) {
		// if we're actually in game, queue up blocks for decoding, so we can
		// spread it out over several frames to avoid introducing lag.
		if(bytes_read < bytes_to_read) {
			adpcm_decode_bytes_read = bytes_read;
			adpcm_decode_bytes_to_read = bytes_to_read;
			adpcm_decode_ptr = ptr;
		}
	} else {
		// if we're in a menu or something, decode the whole buffer now
		while(bytes_read < bytes_to_read) {
			buffer_one_adpcm_block(&bytes_read, bytes_to_read, &ptr);
		}

		//iprintf("Buffering music (%d)\n", bytes_read);
	}
#else
	// check for wrap around
	if(feof(music_fp))
		fseek(music_fp, 0, SEEK_SET);

	// fill in the buffer
	if((bytes_read = fread(ptr, 1, bytes_to_read, music_fp)) < bytes_to_read) {
		ptr += bytes_read;

		fseek(music_fp, 0, SEEK_SET);
		fread(ptr, 1, bytes_to_read - bytes_read, music_fp);
	}

	//iprintf("Buffering music (%d)\n", bytes_read);
#endif

	//DC_FlushRange(ptr, bytes_read);
}

// puts the DS into low-power sleep mode until the ARM7 tells us to wake up
int done_sleeping;
void enter_sleep_mode()
{
	/* save the current IRQs, and turn off everything except the FIFO */
	unsigned long oldIE = REG_IE;
	REG_IE = IRQ_FIFO_NOT_EMPTY;

   	/* tell ARM7 to sleep */
	REG_IPC_FIFO_TX = 0x55443322;

	/* wait until the ARM7 tells us to wake up */
	done_sleeping = 0;
	while(!done_sleeping)
		swiWaitForIRQ();

	/* wait a bit longer until returning power */
	while(REG_VCOUNT != 0);
	while(REG_VCOUNT == 0);
	while(REG_VCOUNT != 0);

	/* restore power, and restore IRQs */
	powerON(POWER_LCD);
	REG_IE = oldIE;
}

// called whenever there's data received on the FIFO (sent from the ARM7)
void FifoHandler(void)
{
	extern int nds_frame_counter;
	int data;

	while ( !(REG_IPC_FIFO_CR & (IPC_FIFO_RECV_EMPTY)) ) {
		data = (int) REG_IPC_FIFO_RX;

		if((u32)data == 0x87654321) {
			Wifi_Sync();
		} else if(data == 0x55443322) {
			done_sleeping = 1;
		} else if(data == 15 && sndsys->channels[15].state != 0) {
			/* This is an IRQ handler, so we want to return quickly.
			 * Defer the actual filling of the buffer to somewhere in the main loop.
			 */
			//buffer_more_music();
			music_needs_rebuffering = true;
		} else {
			iprintf("FIFO Receive: %d\n", data); /* For Debugging from the ARM7 */
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// ROTT SOUND FUNCTIONS
//////////////////////////////////////////////////////////////////////////////////////////////////////////

// Setup the sound card (not much sense in doing anything here, when we could just put it in FX_Init() instead
int FX_SetupCard(int SoundCard, fx_device *device)
{ return FX_Ok; }

// return the current volume
int FX_GetVolume(void)
{ return nds_volume; }

// we've got a hardware volume control, so this is kind of silly, but whatever
void FX_SetVolume(int vol)
{
   	nds_volume = (vol > 255 ? 255 : (vol < 0 ? 0 : vol)) >> 1;
	update_master_volume();
}

// stub; we don't care about error messages
char *FX_ErrorString(int err)
{
   	if(err == FX_Ok)
	   	return "OK";
   	else
	   	return "Error";
}

// Initialize sound system
int FX_Init(int SoundCard, int numvoices, int numchannels, int samplebits, unsigned mixrate)
{ sound_init(); return FX_Ok; }

// stop playback of a particular voice
int FX_StopSound(int handle)
{ channel_stop(handle); return FX_Ok; }

// stop playback of all voices
int FX_StopAllSounds(void)
{ stop_all_sounds(); return FX_Ok; }

// shut down the sound system
int FX_Shutdown(void)
{ FX_StopAllSounds(); return FX_Ok; }

// test if the specified voice is currently playing
int FX_SoundActive(int handle)
{ return !channel_done_playing(handle); }

// set the stereo volume level of the specified voice
int FX_SetPan(int handle, int vol, int left, int right)
{
	int pan;

	if(right == 0)
		pan = 127;
	else
		pan = 64 * left / right;

	if(pan > 127) pan = 127;
	if(pan < 0) pan = 0;

	channel_change_volume(handle, vol >> 1);
	channel_change_pan(handle, pan);

	//iprintf("Panning: L=%d, R=%d, P=%d\n", left, right, pan);
	return FX_Ok;
}

// set the angle and distance from listener of the specified voice
int FX_Pan3D(int handle, int angle, int distance)
{
	int vol, pan;

	calculate_volume_and_panning(angle, distance, &vol, &pan);

	channel_change_volume(handle, vol >> 1);
	channel_change_pan(handle, pan);

	return FX_Ok;
}

// set the pitch of the specified voice
int FX_SetPitch(int handle, int pitchoffset)
{ return FX_Ok; }

// test if a voice can be played at the specified priority
int FX_VoiceAvailable(int priority)
{ return find_free_channel(priority) != -1; }

// sets the function to call when a voice is done playing
int FX_SetCallBack(void ( *function )( unsigned long ))
{ return FX_Ok; }

// plays raw sound data
static inline int NDS_PlayRaw(char *buf, int length, int pitchoffset, int angle, int distance,
		int priority, unsigned long callbackval, int sample_rate, int format)
{
//#define LOG_RAW_SOUNDS
#ifdef LOG_RAW_SOUNDS
	static int i = 0;
	char filename[12];
	FILE *fp;
	snprintf(filename, 12, "sound%d.raw", i++); filename[11] = '\0';
	fp = fopen(filename, "w");
	if(fp != NULL) {
		if(fwrite(buf, length, 1, fp) != 1) {
			fclose(fp);
			unlink(fp);
		} else {
			fclose(fp);
		}
	}
#endif

	int vol, pan;
	int channel = find_free_channel(priority);
	if(channel > -1) {
		//iprintf("Playing on chan %d, len=%d\n", channel, length);
		calculate_volume_and_panning(angle, distance, &vol, &pan);
		channel_start_sound(channel, buf, length, vol, pan, 0, sample_rate, (format == 1 ? 8 : 16), priority);
	} else {
		//iprintf("No channel available\n");
	}

	//setGenericSound(sample_rate, vol, pan, format);
	//playGenericSound(buf, length);

	return FX_Ok;
}

// these structs are for decoding a VOC sound file (Sound Blaster)
typedef struct {
	char description[20];
	short data_start;
	short version;
	short id_code;
} voc_header;

typedef struct {
	char block_type;
	char block_length[3];
} voc_block_start;

typedef struct {
	short sample_rate;
	char format;
	char is_stereo;
} voc_data_block8;

typedef struct {
	unsigned sample_rate;
	char bitrate;
	char channel;
	unsigned short format;
	int reserved;
} voc_data_block9;

// begin playback of sound data at a specified angle and distance from listener (VOC)
int FX_PlayVOC3D(char *raw_ptr, int pitchoffset, int angle, int distance,
       int priority, unsigned long callbackval)
{
	static unsigned char buffer[1024*32]; // I don't think ROTT has any sound effects larger than 32KB
	unsigned char *ptr, *buf;
	voc_header *head;
	voc_block_start *block1;
	voc_data_block9 *block9;
	voc_data_block8 *block8;
	int block_length, sample_rate, format, bitrate, channels, is_signed;
	unsigned short repeat = 0;
	int done = 0, extended = 0, bytes_to_repeat = 0;

	ptr = (unsigned char*) raw_ptr;
	buf = buffer;

	//iprintf("FX_PlayVOC3D()\n");

	head = (voc_header*) ptr;
	if(strncmp(head->description, "Creative Voice File", 19) != 0 || (~head->version + 0x1234 != head->id_code)) {
		iprintf("BADSOUND: Not a valid VOC file\n");
		return FX_Error;
	}
	ptr += head->data_start;

	/* Note: this code makes the assumption that all blocks in a given VOC file use the same parameters
	 * (sample rate, channels, bitrate, etc.).  This is probably safe, at least for the sound effects in ROTT.
	 */
	while(!done) {
		block1 = (voc_block_start*) ptr;
		block_length = block1->block_length[0] | (block1->block_length[1] << 8) | (block1->block_length[2] << 16);
		ptr += sizeof(voc_block_start);
		//iprintf("block %d, length = %d\n", block1->block_type, block_length);

		switch(block1->block_type) {
			case 0: /* Terminator */
				done = 1;
				break;
			case 1: /* Normal Data Block */
				if(extended) {
					// use values from the previous block type 8 instead.
					ptr += 2; // skip past the sample rate and format bytes.
				} else {
					sample_rate = -1000000 / (*ptr++ - 256);
					format = *ptr++; // 0 => 8bit unsigned; 1 => 4bit ADPCM, 2 => 2.6bit ADPCM; 3 => 2bit ADPCM
					bitrate = 8;
					channels = 1;
				}

				block_length -= 2; // don't count sample_rate and format bytes
				extended = 0;

				if(format == 0) {
					format = 1; // 8-bit
					is_signed = 0;
				} else if(format == 1 || format == 2 || format == 3) {
					iprintf("BADSOUND1: ADPCM format %d\n", format);
					return FX_Error; // don't think this is used by ROTT
				} else {
					iprintf("BADSOUND1: unknown format %d\n", format);
					return FX_Error;
				}

				if(buf + block_length > buffer + 32 * 1024) {
					memcpy(buf, ptr, buffer + 32 * 1024 - buf);
					if(!is_signed) unsigned_to_signed(buf, buffer + 32 * 1024 - buf);
					done = 1;
					break;
				}
				memcpy(buf, ptr, block_length);
				if(!is_signed) unsigned_to_signed(buf, block_length);
				buf += block_length;
				ptr += block_length;
				if(repeat > 0) bytes_to_repeat += block_length;
				break;
			case 9: /* New Fancy Data Block */
				block9 = (voc_data_block9*) ptr;
				ptr += sizeof(voc_data_block9);
				sample_rate = block9->sample_rate;
				bitrate = block9->bitrate;
				channels = block9->channel;
				format = block9->format; // 0 => unsigned, 1,2,3,0x200 => ADPCM, 4 => signed, 6 => alaw, 7 => mulaw
				block_length -= sizeof(voc_data_block9); // don't count block header

				if(format == 1 || format == 2 || format == 3 || format == 0x200) {
					iprintf("BADSOUND9: ADPCM format %d\n", format);
					return FX_Error; // don't think this is used by ROTT
				} else if(format == 0) {
					format = (bitrate == 16 ? 0 : 1);
					is_signed = 0;
				} else if(format == 4) {
					format = (bitrate == 16 ? 0 : 1);
					is_signed = 1;
				} else {
					iprintf("BADSOUND9: unknown format %d\n", format);
					return FX_Error; // don't think alaw or mulaw are used by ROTT
				}

				if(buf + block_length > buffer + 32 * 1024) {
					memcpy(buf, ptr, buffer + 32 * 1024 - buf);
					if(!is_signed) unsigned_to_signed(buf, buffer + 32 * 1024 - buf);
					done = 1;
					break;
				}
				memcpy(buf, ptr, block_length);
				if(!is_signed) unsigned_to_signed(buf, block_length);
				buf += block_length;
				ptr += block_length;
				if(repeat > 0) bytes_to_repeat += block_length;
				break;
			case 8: /* Extended Block */
				block8 = (voc_data_block8*) ptr;
				ptr += sizeof(voc_data_block8);
				format = block8->format;
				channels = (block8->is_stereo ? 2 : 1);
				sample_rate = (256000000L / (65536L - block8->sample_rate)) / channels;
				block_length -= sizeof(voc_data_block8);
				ptr += block_length;
				extended = 1;
				// Note: a block type 1 always follows a block type 8
				break;
			case 6: /* Loop Start Block */
				repeat = *(unsigned short*)ptr + 1;
				ptr += sizeof(unsigned short);
				bytes_to_repeat = 0;
				break;
			case 7: /* Loop End Block */
				while(repeat > 0) {
					if(buf + bytes_to_repeat > buffer + 32 * 1024) {
						memcpy(buf, buf - bytes_to_repeat, buffer + 32 * 1024 - buf);
						if(!is_signed) unsigned_to_signed(buf, buffer + 32 * 1024 - buf);
						done = 1;
						break;
					}
					memcpy(buf, buf - bytes_to_repeat, bytes_to_repeat);
					if(!is_signed) unsigned_to_signed(buf, bytes_to_repeat);
					buf += bytes_to_repeat;
					repeat--;
				}
				break;
			default:
				// we can safely ignore other block types in most cases
				//iprintf("Unknown VOC block type: %d\n", block1->block_type);
				ptr += block_length;
		}
	}

	//iprintf("Playing a VOC at %d Hz, %s, %d bit (%d bytes)\n", sample_rate, channels == 2 ? "Stereo" : "Mono", bitrate, ((int)buf - (int)buffer));
	return NDS_PlayRaw(buffer, ((int)buf - (int)buffer), pitchoffset, angle, distance, priority, callbackval, sample_rate, format);
}

// these structs are for decoding a standard WAVE file
typedef struct {
	char group_id[4];
	int riff_size;
	char riff_type[4];
} riff_header;

typedef struct {
	char chunk_id[4];
	int chunk_size;
	short format_tag;
	unsigned short channels;
	unsigned sample_rate;
	unsigned avg_bytes_per_sec;
	unsigned short block_align;
	unsigned short bitrate;
} riff_format_chunk;

// begin playback of sound data at a specified angle and distance from listener (WAV)
int FX_PlayWAV3D(char *raw_ptr, int pitchoffset, int angle, int distance,
       int priority, unsigned long callbackval)
{
	/* XXX: I don't think ROTT even uses any WAV sounds, but the code can potentially
	 * call this function, and basic WAV support is easy enough, so I'll go ahead and implement it */

	static unsigned char buffer[1024*32]; // I don't think ROTT has any sound effects larger than 32KB
	unsigned char *ptr, *buf;
	riff_header *head;
	riff_format_chunk *format_chunk;
	int chunk_size, sample_rate, channels, bitrate, format;

	ptr = (unsigned char*) raw_ptr;
	buf = buffer;

	//iprintf("FX_PlayWAV3D()\n");

	head = (riff_header*) ptr;
	if(strncmp(head->group_id, "RIFF", 4) != 0 || strncmp(head->riff_type, "WAVE", 4) != 0)
		return FX_Error;

	ptr += sizeof(riff_header);

	format_chunk = (riff_format_chunk*) ptr; // 0 => no compression; otherwise defines compression rate
	sample_rate = format_chunk->sample_rate;
	bitrate = format_chunk->bitrate;
	channels = format_chunk->channels;
	format = format_chunk->format_tag;
	ptr += sizeof(riff_format_chunk);

	if(strncmp(format_chunk->chunk_id, "fmt ", 4) != 0 || format != 1)
		return FX_Error;

	if(strncmp(ptr, "data", 4) != 0)
		return FX_Error;
	ptr += 4;

	chunk_size = *(int*)ptr;
	ptr += 4;

	memcpy(buf, ptr, chunk_size);

	//iprintf("Playing a WAV at %d Hz, %s, %d bit\n", sample_rate, channels == 2 ? "Stereo" : "Mono", bitrate);
	format = (bitrate == 16 ? 0 : 1);
	return NDS_PlayRaw(buffer, chunk_size, pitchoffset, angle, distance, priority, callbackval, sample_rate, format);
}

// play a digitized sound from a user-controlled buffer
int FX_StartDemandFeedPlayback(void ( *function )( char **ptr, unsigned long *length ),
       int rate, int pitchoffset, int vol, int left, int right,
       int priority, unsigned long callbackval)
{
	// stub: this is only used in network games
	return FX_Ok;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// ROTT MUSIC FUNCTIONS
//////////////////////////////////////////////////////////////////////////////////////////////////////////

// stub; we don't care about error messages
char *MUSIC_ErrorString(int ErrorNumber)
{
	if(ErrorNumber == MUSIC_Ok)
		return "OK";
	else
		return "Error";
}

// stub: initialize music
int MUSIC_Init(int SoundCard, int Address)
{ return MUSIC_Ok; }

// shutdown music
int MUSIC_Shutdown(void)
{
   	MUSIC_StopSong();
   	return MUSIC_Ok;
}

// sets music volume
void MUSIC_SetVolume(int vol)
{
   	nds_music_volume = (vol > 255 ? 255 : (vol < 0 ? 0 : vol)) >> 1;

	if(MUSIC_SongPlaying()) {
		if(nds_music_volume == 0)
			MUSIC_StopSong(); // Note: music won't re-start when you increase the volume again
		else
			channel_change_volume(15, nds_music_volume);
	}
}

// returns music volume
int MUSIC_GetVolume(void)
{ return nds_music_volume; }

// test if music is currently playing
int MUSIC_SongPlaying(void)
{
	if(sndsys->channels[15].state == 0)
		return 0;
	else
		return 1;
}

// continue music playback after a pause
void MUSIC_Continue(void)
{ /* stub */ }

// pause music playback
void MUSIC_Pause(void)
{ /* stub */ }

// stop playing music
int MUSIC_StopSong(void)
{
	channel_stop(15);

	if(music_fp) {
		fclose(music_fp);
		music_fp = NULL;
	}

   	return MUSIC_Ok;
}

// Start playing a music file (either 'song' is a pointer to some ADPCM data,
// or it's NULL and 'music_fp' is an open file handle ready to read an ADPCM file.
int MUSIC_PlaySongROTT(unsigned char *song, int size, int loopflag)
{
	//while(MUSIC_SongPlaying()); // wait for the ARM7 to stop playing the current song, if any

	if(nds_music_volume > 0) {
		if(song) {
			// music fully loaded into RAM; play it like an ordinary sound file
			channel_start_sound(15, song, size, nds_music_volume, 64, loopflag, MUSIC_SAMPLE_RATE, 255, 10);
		} else if(music_fp) {
			// stream music from disk
#ifndef NO_SOFTWARE_ADPCM_DECODING
			music_buffer_len = 0;
			while(music_buffer_len + size * 2 < MAX_BUFFER_SIZE)
				music_buffer_len += size * 2; // make our buffer size an even multiple of the ADPCM block size

			buffer_more_music();

			channel_start_sound(15, music_ptr, music_buffer_len, nds_music_volume, 64, loopflag, MUSIC_SAMPLE_RATE, 16, -10);
#else
			channel_start_sound(15, music_ptr, music_buffer_len, nds_music_volume, 64, loopflag, MUSIC_SAMPLE_RATE, 255, -10);
#endif
		}
	}

   	return MUSIC_Ok;
}

// stub
void MUSIC_SetSongTick(unsigned long PositionInTicks)
{ }

// stub
void MUSIC_SetSongTime(unsigned long milliseconds)
{ }

// stub
void MUSIC_SetSongPosition(int measure, int beat, int tick)
{ }

// stub
void MUSIC_GetSongPosition(songposition *pos)
{ }

// stub
void MUSIC_GetSongLength(songposition *pos)
{ }

// stub: start fading out music
int MUSIC_FadeVolume(int tovolume, int milliseconds)
{ return MUSIC_Ok; }

// stub: tests if music is fading
int MUSIC_FadeActive(void)
{ return 0; }

// stub: stop music fading
void MUSIC_StopFade(void)
{ }

