/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggTheora SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE Theora SOURCE CODE IS COPYRIGHT (C) 2002-2003                *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

  function:
  last mod: $Id: theora.h,v 1.17 2003/12/06 18:06:19 arc Exp $

 ********************************************************************/

#ifndef _O_THEORA_H_
#define _O_THEORA_H_

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

#include "ogg.h"

typedef struct {
    int   y_width;      /**< Width of the Y' luminance plane */
    int   y_height;     /**< Height of the luminance plane */
    int   y_stride;     /**< Offset in bytes between successive rows */

    int   uv_width;     /**< Height of the Cb and Cr chroma planes */
    int   uv_height;    /**< Width of the chroma planes */
    int   uv_stride;    /**< Offset between successive chroma rows */
    unsigned char *y;   /**< Pointer to start of luminance data */
    unsigned char *u;   /**< Pointer to start of Cb data */
    unsigned char *v;   /**< Pointer to start of Cr data */

} yuv_buffer;

typedef enum {
  OC_CS_UNSPECIFIED,    /**< The colorspace is unknown or unspecified */
  OC_CS_ITU_REC_470M,   /**< This is the best option for 'NTSC' content */
  OC_CS_ITU_REC_470BG,  /**< This is the best option for 'PAL' content */
  OC_CS_NSPACES         /**< This marks the end of the defined colorspaces */
} theora_colorspace;

typedef enum {
  OC_PF_420,    /**< Chroma subsampling by 2 in each direction (4:2:0) */
  OC_PF_RSVD,   /**< Reserved value */
  OC_PF_422,    /**< Horizonatal chroma subsampling by 2 (4:2:2) */
  OC_PF_444,    /**< No chroma subsampling at all (4:4:4) */
} theora_pixelformat;

typedef struct {
  ogg_uint32_t  width;		/**< encoded frame width  */
  ogg_uint32_t  height;		/**< encoded frame height */
  ogg_uint32_t  frame_width;	/**< display frame width  */
  ogg_uint32_t  frame_height;	/**< display frame height */
  ogg_uint32_t  offset_x;	/**< horizontal offset of the displayed frame */
  ogg_uint32_t  offset_y;	/**< vertical offset of the displayed frame */
  ogg_uint32_t  fps_numerator;	    /**< frame rate numerator **/
  ogg_uint32_t  fps_denominator;    /**< frame rate denominator **/
  ogg_uint32_t  aspect_numerator;   /**< pixel aspect ratio numerator */
  ogg_uint32_t  aspect_denominator; /**< pixel aspect ratio denominator */
  theora_colorspace colorspace;	    /**< colorspace */
  int           target_bitrate;	    /**< nominal bitrate in bits per second */
  int           quality;  /**< Nominal quality setting, 0-63 */
  int           quick_p;  /**< Quick encode/decode */

  /* decode only */
  unsigned char version_major;
  unsigned char version_minor;
  unsigned char version_subminor;

  void *codec_setup;

  /* encode only */
  int           dropframes_p;
  int           keyframe_auto_p;
  ogg_uint32_t  keyframe_frequency;
  ogg_uint32_t  keyframe_frequency_force;  /* also used for decode init to
                                              get granpos shift correct */
  ogg_uint32_t  keyframe_data_target_bitrate;
  ogg_int32_t   keyframe_auto_threshold;
  ogg_uint32_t  keyframe_mindistance;
  ogg_int32_t   noise_sensitivity;
  ogg_int32_t   sharpness;

  theora_pixelformat pixelformat;	/**< chroma subsampling mode to expect */

} theora_info;

/** Codec internal state and context.
 */
typedef struct{
  theora_info *i;
  ogg_int64_t granulepos;

  void *internal_encode;
  void *internal_decode;

} theora_state;

typedef struct theora_comment{
  char **user_comments;         /**< An array of comment string vectors */
  int   *comment_lengths;       /**< An array of corresponding string vector lengths in bytes */
  int    comments;              /**< The total number of comment string vectors */
  char  *vendor;                /**< The vendor string identifying the encoder, null terminated */

} theora_comment;

#define OC_FAULT       -1       /**< General failure */
#define OC_EINVAL      -10      /**< Library encountered invalid internal data */
#define OC_DISABLED    -11      /**< Requested action is disabled */
#define OC_BADHEADER   -20      /**< Header packet was corrupt/invalid */
#define OC_NOTFORMAT   -21      /**< Packet is not a theora packet */
#define OC_VERSION     -22      /**< Bitstream version is not handled */
#define OC_IMPL        -23      /**< Feature or action not implemented */
#define OC_BADPACKET   -24      /**< Packet is corrupt */
#define OC_NEWPACKET   -25      /**< Packet is an (ignorable) unhandled extension */
#define OC_DUPFRAME    1        /**< Packet is a dropped frame */

extern const char *theora_version_string(void);

extern ogg_uint32_t theora_version_number(void);

extern int theora_encode_init(theora_state *th, theora_info *ti);

extern int theora_encode_YUVin(theora_state *t, yuv_buffer *yuv);

extern int theora_encode_packetout( theora_state *t, int last_p,
                                    ogg_packet *op);

extern int theora_encode_header(theora_state *t, ogg_packet *op);

extern int theora_encode_comment(theora_comment *tc, ogg_packet *op);

extern int theora_encode_tables(theora_state *t, ogg_packet *op);

extern int theora_decode_header(theora_info *ci, theora_comment *cc,
                                ogg_packet *op);

extern int theora_decode_init(theora_state *th, theora_info *c);

extern int theora_decode_packetin(theora_state *th,ogg_packet *op);

extern int theora_decode_YUVout(theora_state *th,yuv_buffer *yuv);

extern int theora_packet_isheader(ogg_packet *op);

extern int theora_packet_iskeyframe(ogg_packet *op);

int theora_granule_shift(theora_info *ti);

extern ogg_int64_t theora_granule_frame(theora_state *th,ogg_int64_t granulepos);

extern double theora_granule_time(theora_state *th,ogg_int64_t granulepos);

extern void theora_info_init(theora_info *c);

extern void theora_info_clear(theora_info *c);

extern void theora_clear(theora_state *t);

extern void theora_comment_init(theora_comment *tc);

extern void theora_comment_add(theora_comment *tc, char *comment);

extern void theora_comment_add_tag(theora_comment *tc,
                                       char *tag, char *value);

extern char *theora_comment_query(theora_comment *tc, char *tag, int count);

extern int   theora_comment_query_count(theora_comment *tc, char *tag);

extern void  theora_comment_clear(theora_comment *tc);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* _O_THEORA_H_ */
