#include <nds.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "graphics.h"
#include "objects.h"

Picture extras_pic, bill_pic, os_pic, swap_pic, systems_pic;

extern SpriteEntry sprites[128];
extern pSpriteRotation sprite_rot;

u16 *bg0map, *bg1map, *bg2map;
u8 *bg3map;
int can_double_buffer = 0;

int main_screen_row = 0;
int main_screen_col = 0;

void double_buffer_init()
{
	bg0map = (u16*)malloc(32 * 24 * 2);
	if(bg0map == NULL) {
		bg0map = (u16*)SCREEN_BASE_BLOCK(0);
		bg1map = (u16*)SCREEN_BASE_BLOCK(2);
		bg2map = (u16*)SCREEN_BASE_BLOCK(4);
		bg3map = (u8*)BG_BMP_RAM(3);
		can_double_buffer = 0;
		return;
	}

	bg1map = (u16*)malloc(32 * 24 * 2);
	if(bg1map == NULL) {
		free(bg0map);
		bg0map = (u16*)SCREEN_BASE_BLOCK(0);
		bg1map = (u16*)SCREEN_BASE_BLOCK(2);
		bg2map = (u16*)SCREEN_BASE_BLOCK(4);
		bg3map = (u8*)BG_BMP_RAM(3);
		can_double_buffer = 0;
		return;
	}

	bg2map = (u16*)malloc(32 * 24 * 2);
	if(bg2map == NULL) {
		free(bg0map);
		free(bg1map);
		bg0map = (u16*)SCREEN_BASE_BLOCK(0);
		bg1map = (u16*)SCREEN_BASE_BLOCK(2);
		bg2map = (u16*)SCREEN_BASE_BLOCK(4);
		bg3map = (u8*)BG_BMP_RAM(3);
		can_double_buffer = 0;
		return;
	}

	bg3map = (u8*)malloc(256 * 192);
	if(bg3map == NULL) {
		free(bg0map);
		free(bg1map);
		free(bg2map);
		bg0map = (u16*)SCREEN_BASE_BLOCK(0);
		bg1map = (u16*)SCREEN_BASE_BLOCK(2);
		bg2map = (u16*)SCREEN_BASE_BLOCK(4);
		bg3map = (u8*)BG_BMP_RAM(3);
		can_double_buffer = 0;
		return;
	}

	can_double_buffer = 1;
}

void double_buffer()
{
	if(can_double_buffer) {
		DC_FlushRange(bg0map, 32 * 24 * 2);
		dmaCopy(bg0map, (u16*)SCREEN_BASE_BLOCK(0), 32 * 24 * 2);
		DC_FlushRange(bg1map, 32 * 24 * 2);
		dmaCopy(bg1map, (u16*)SCREEN_BASE_BLOCK(2), 32 * 24 * 2);
		DC_FlushRange(bg2map, 32 * 24 * 2);
		dmaCopy(bg2map, (u16*)SCREEN_BASE_BLOCK(4), 32 * 24 * 2);
		DC_FlushRange(bg3map, 256 * 192);
		dmaCopy(bg3map, (u16*)BG_BMP_RAM(3), 256 * 192);
	}
}

void Win2DrawChars(char *chars, int len, short x, short y)
{
	iprintf("\x1b[%d;%dH%s", y / 8, x / 8, chars);
}

#define sign(x) (((x) > 0) ? 1 : (((x) < 0) ? -1 : 0))
void Win2DrawLine(short x1, short y1, short x2, short y2)
{
	// Bresenham's algorith
	int x = x1;
   	int y = y1;
	int dx = abs(x2 - x1);
	int	dy = abs(y2 - y1);
	int s1 = sign(x2 - x1);
	int s2 = sign(y2 - y1);
	int swap = 0;

	if(dy > dx) {
		int temp = dx;
		dx = dy;
		dy = temp;
		swap = 1;
	}

	int D = 2 * dy - dx;

	for(int i = 0; i < dx; i++) {
		bg3map[y * 256 + x] = 255;

		while (D >= 0) {
			D = D - 2 * dx;
			if(swap)
				x += s1;
			else
				y += s2;
		} 

		D = D + 2 * dy;

		if(swap)
			y += s2;
		else
			x += s1; 
	}
}

void Win2ClearScreen()
{
	int i;

	for(i = 0; i < 32 * 24; i++) {
		bg0map[i] = 0;
		bg1map[i] = 0;
		bg2map[i] = (15 << 12) | ' ';
	}

	memset(bg3map, 0, 256 * 192);

	main_screen_col = 0;
	main_screen_row = 0;
}

void draw_image(Picture pict, int pic_index, int x, int y, int sprite)
{
	switch(pict.pic_type) {
		case PIC_BILL:
			draw_bill(x, y, pic_index, sprite);
			break;
		case PIC_EXTRAS:
			draw_extras(x, y, pic_index, sprite);
			break;
		case PIC_SWAP:
			draw_swap(x, y, pic_index, sprite);
			break;
		case PIC_OS:
			draw_os(x, y, pic_index);
			break;
		case PIC_SYSTEMS:
			draw_system(x, y, pic_index);
			break;
	}
}

void remove_sprite(int index)
{
	sprites[index].attribute[0] = ATTR0_DISABLED;
}

void draw_bill(int x, int y, int sprite, int index)
{
	// 16x32 (really 16x24; bottom 8 rows are transparent)
	if(sprite < 0) sprite = 0; else if(sprite >= 11) sprite = 11;

	if(x >= 0 && x < 256 - 16 && y >= 0 && y < 192 - 24) {
		sprites[index].attribute[0] = ATTR0_COLOR_256 | ATTR0_TALL | (y & 0xFF);
		sprites[index].attribute[1] = ATTR1_SIZE_32 | (x & 0x1FF);
		sprites[index].attribute[2] = ATTR2_PRIORITY(1) | ((sprite) * 16);
	}
}

void draw_extras(int x, int y, int sprite, int index)
{
	// 16x16
	if(sprite < 0) sprite = 0; else if(sprite >= 3) sprite = 3;

	if(x >= 0 && x < 256 - 16 && y >= 0 && y < 192 - 16) {
		sprites[index].attribute[0] = ATTR0_COLOR_256 | ATTR0_SQUARE | (y & 0xFF);
		sprites[index].attribute[1] = ATTR1_SIZE_16 | (x & 0x1FF);
		sprites[index].attribute[2] = ATTR2_PRIORITY(0) | (176 + (sprite) * 8);
	}
}

void draw_swap(int x, int y, int sprite, int index)
{
	// 32x32 (really 32x24; bottom 8 rows are transparent)
	if(sprite < 0) sprite = 0; else if(sprite >= 13) sprite = 13;

	if(x >= 0 && x < 256 - 32 && y >= 0 && y < 192 - 24) {
		sprites[index].attribute[0] = ATTR0_COLOR_256 | ATTR0_SQUARE | (y & 0xFF);
		sprites[index].attribute[1] = ATTR1_SIZE_32 | (x & 0x1FF);
		sprites[index].attribute[2] = ATTR2_PRIORITY(1) | (176 + 24 + (sprite) * 32);
	}
}

void draw_system(int x, int y, int sprite)
{
	// 32x24
	if(sprite < 0) sprite = 0; else if(sprite >= 9) sprite = 9;

	x /= 8;
	y /= 8;

	bg1map[y * 32 + x] = (sprite + 1) * 4 - 3;
	bg1map[y * 32 + x + 1] = (sprite + 1) * 4 + 1 - 3;
	bg1map[y * 32 + x + 2] = (sprite + 1) * 4 + 2 - 3;
	bg1map[y * 32 + x + 3] = (sprite + 1) * 4 + 3 - 3;
	bg1map[(y + 1) * 32 + x] = 36 + (sprite + 1) * 4 - 3;
	bg1map[(y + 1) * 32 + x + 1] = 36 + (sprite + 1) * 4 + 1 - 3;
	bg1map[(y + 1) * 32 + x + 2] = 36 + (sprite + 1) * 4 + 2 - 3;
	bg1map[(y + 1) * 32 + x + 3] = 36 + (sprite + 1) * 4 + 3 - 3;
	bg1map[(y + 2) * 32 + x] = 72 + (sprite + 1) * 4 - 3;
	bg1map[(y + 2) * 32 + x + 1] = 72 + (sprite + 1) * 4 + 1 - 3;
	bg1map[(y + 2) * 32 + x + 2] = 72 + (sprite + 1) * 4 + 2 - 3;
	bg1map[(y + 2) * 32 + x + 3] = 72 + (sprite + 1) * 4 + 3 - 3;
}

void draw_os(int x, int y, int sprite)
{
	// 16x16
	if(sprite < 0) sprite = 0; else if(sprite >= 12) sprite = 12;

	x /= 8;
	y /= 8;

	if(x >= 0 && y >= 0 && x < 32 && y < 24)
		bg0map[y * 32 + x] = 108 + (sprite + 1) * 2 - 1;
	if(x + 1 >= 0 && y >= 0 && x + 1 < 32 && y < 24)
		bg0map[y * 32 + x + 1] = 108 + (sprite + 1) * 2 + 1 - 1;
	if(x >= 0 && y + 1 >= 0 && x < 32 && y + 1 < 24)
		bg0map[(y + 1) * 32 + x] = 108 + 24 + (sprite + 1) * 2 - 1;
	if(x + 1 >= 0 && y + 1 >= 0 && x + 1 < 32 && y + 1 < 24)
		bg0map[(y + 1) * 32 + x + 1] = 108 + 24 + (sprite + 1) * 2 + 1 - 1;
}

static u32 spark_sprites = 0;
int find_available_sprite()
{
	int i;
	for(i = 0; i < 25; i++) {
		if(!(spark_sprites & BIT(i))) {
			spark_sprites |= BIT(i);
			return 125 - i;
		}
	}
	return 0;
}

void done_with_sprite(int index)
{
	spark_sprites &= ~BIT(125 - index);
}

void load_bitmap(const unsigned *src, int len)
{
	ui.clear();
	dmaCopy(src, bg3map, len);
	ui.refresh();
}

void main_screen_print_char(int c)
{
	if(main_screen_row >= 24) return;
	if(c == '\n') {
		main_screen_col = 0;
		main_screen_row++;
	} else {
		bg2map[main_screen_col + main_screen_row * 32] = (15 << 12) | (u16)c;

		main_screen_col++;
		if(main_screen_col >= 32) {
			main_screen_col = 0;
			main_screen_row++;
		}
	}
}

void main_screen_print(char *str)
{
	while(*str != '\0') {
		main_screen_print_char(*str);
		str++;
	}
}

void main_screen_printf(char *fmt, ...)
{
	static char buf[1024];
	va_list args;

	va_start(args, fmt);
	vsnprintf(buf, 1024, fmt, args); buf[1023] = '\0';
	va_end(args);

	main_screen_print(buf);
}

void main_screen_gotoxy(int col, int row)
{
	if(col >= 0 && col < 32)
		main_screen_col = col;
	if(row >= 0 && row < 24)
		main_screen_row = row;
}
